/*	MiniHealthHUDItemClass
	Copyright 2008 Mark Sararu

	This file is part of the Renegade scripts.dll
	The Renegade scripts.dll is free software; you can redistribute it and/or modify it under
	the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2, or (at your option) any later
	version. See the file COPYING for more details.
	In addition, an exemption is given to allow Run Time Dynamic Linking of this code with any closed source module 
	that does not contain code covered by this licence.
	Only the source code to the module(s) containing the licenced code has to be released.
*/

#include "scripts.h"
#include "shadereng.h"
#include "minihealthhud.h"

//--------------------------------------------------------------------------------
// helper functions
//--------------------------------------------------------------------------------
class PIXDebugEventRAII
{
public:
	PIXDebugEventRAII(D3DCOLOR color, const char * file, const char * function)
	{
		DebugEventStart(color, L"%S!%S", file, function);
	};
	~PIXDebugEventRAII()
	{
		DebugEventEnd();
	};
};

#define DEBUG_INSTRUMENT_FUNCTION PIXDebugEventRAII PIXDebugEvent##__COUNTER__( DEBUG_COLOR1 , __FILE__ , __FUNCTION__ ) 

D3DCOLOR Vector4ToD3DCOLOR(Vector4& color)
{
	return D3DCOLOR_ARGB(char(color.W * 255), char(color.X * 255), char(color.Y * 255), char(color.Z * 255));
};

//--------------------------------------------------------------------------------
// health hud item rendering/loading code
//--------------------------------------------------------------------------------
MiniHealthHUDItemClass::MiniHealthHUDItemClass(): 
	Enabled(false),
	Render2D(NULL), 
	Render2DText(NULL),
	LastHealth(0.0f),
	FlashActive(false),
	EndFlashTime(0),
	EndBlinkTime(0),
	ForceInfoDisplayDuringLowHealth(true),
	LowHealthThreshold(50.0f),
	HealthInfoFlashDuration(5000),
	LowHealthBlinkDuration(500),
	BackgroundSize(0, 0),
	BackgroundPosition(0, 0),
	BackgroundUV(0, 0, 0, 0),
	BackgroundTextureFile(NULL),
	TextPosition(0, 0),
	TextFontFile(NULL)
{

};

MiniHealthHUDItemClass::~MiniHealthHUDItemClass()
{
	SAFE_DELETE(Render2D);
	SAFE_DELETE(Render2DText);
	SAFE_DELETE(BackgroundTextureFile);
	SAFE_DELETE(TextFontFile);
};

void MiniHealthHUDItemClass::Load(INIClass *ini)
{
	if (!ini) return; // if you don't have an ini, something is horribly wrong!

	const char* section_name = "MiniHealthDisplay";

	Enabled = ini->Get_Bool(section_name, "Enabled", false);
	if (!Enabled) return;

	Render2D = CreateRender2DClass();

	BackgroundSize.X = ini->Get_Float(section_name, "Background.Width", 0.0f);
	BackgroundSize.Y = ini->Get_Float(section_name, "Background.Height", 0.0f);

	char temp[512];
	ini->Get_String(section_name, "Background.Texture.File", "DEFAULT", temp, 512);
	BackgroundTextureFile = newstr(temp);

	RectClass background_texture_quad;
	background_texture_quad.Left = ini->Get_Float(section_name, "Background.Texture.Quad.Left", 0.0f);
	background_texture_quad.Top = ini->Get_Float(section_name, "Background.Texture.Quad.Top", 0.0f);
	background_texture_quad.Right = ini->Get_Float(section_name, "Background.Texture.Quad.Right", background_texture_quad.Left + BackgroundSize.X);
	background_texture_quad.Bottom = ini->Get_Float(section_name, "Background.Texture.Quad.Bottom", background_texture_quad.Top + BackgroundSize.Y);

	Render2D->Set_Texture(BackgroundTextureFile);
	if (Render2D->Texture->Initialized == false)
	{
		Render2D->Texture->Init();
	};

	float tex_width = float(1.0f / Render2D->Texture->Width);
	float tex_height = float(1.0f / Render2D->Texture->Height);

	BackgroundUV.Left = background_texture_quad.Left * tex_width;
	BackgroundUV.Top = background_texture_quad.Top * tex_height;
	BackgroundUV.Right = background_texture_quad.Right * tex_width;
	BackgroundUV.Bottom = background_texture_quad.Bottom * tex_height;

	bool background_centered = ini->Get_Bool(section_name, "Background.Position.Centered", true);
	BackgroundPosition.X = ini->Get_Float(section_name, "Background.Position.X", -200.0f);
	BackgroundPosition.Y = ini->Get_Float(section_name, "Background.Position.Y", 0.0f);

	Vector2 screen_center;
	screen_center.X = (ScreenResolution->Right - ScreenResolution->Left) / 2.0f;
	screen_center.Y = (ScreenResolution->Bottom - ScreenResolution->Top) / 2.0f;

	if (background_centered)
	{
		BackgroundPosition = BackgroundPosition + screen_center - (BackgroundSize / 2.0f) ;
	}

	ini->Get_String(section_name, "Text.Font.File", "DEFAULT_FONT", temp, 512);
	Render2DText = CreateRender2DTextClass(temp);
	TextFontFile = newstr(temp);

	float average_height = ini->Get_Float(section_name, "Text.Font.AverageCharacterHeight", 16);

	bool text_centered = ini->Get_Bool(section_name, "Text.Position.Centered", true);
	TextPosition.X = ini->Get_Float(section_name, "Text.Position.X", -195.0f);
	TextPosition.Y = ini->Get_Float(section_name, "Text.Position.Y", 0.0f);
	if (text_centered)
	{
		TextPosition = TextPosition + screen_center;
		TextPosition.Y -= average_height / 2.0f;
	}

	ForceInfoDisplayDuringLowHealth = ini->Get_Bool(section_name, "ForceInfoDisplayDuringLowHealth", true);
	LowHealthThreshold = ini->Get_Float(section_name, "LowHealthThreshold", 50.0f);

	HealthInfoFlashDuration = ini->Get_Int(section_name, "HealthInfoFlashDuration", 5000);
	LowHealthBlinkDuration = ini->Get_Int(section_name, "LowHealthBlinkDuration", 500);
};

void MiniHealthHUDItemClass::Render()
{
	DEBUG_INSTRUMENT_FUNCTION;

	if (!Enabled) return;

	GameObject *obj = Get_Vehicle_Return((GameObject *)(*TheStar)->obj);
	float current_health = Commands->Get_Health(obj);
	unsigned int current_time = *SyncTime;

	if (current_health != LastHealth && (current_health > LowHealthThreshold && ForceInfoDisplayDuringLowHealth))
	{
		EndFlashTime = current_time + HealthInfoFlashDuration;
		FlashActive = true;
	}
	else if ((current_health <= LowHealthThreshold) && ForceInfoDisplayDuringLowHealth)
	{
		FlashActive = false;
	}

	float flash_percentage = 1.0f;
	if (FlashActive && (current_time < EndFlashTime))
	{
		flash_percentage = float(EndFlashTime - current_time) / float(HealthInfoFlashDuration);
	}
	else if (FlashActive)
	{
		flash_percentage = 0.0f;
	}

	float blink_percentage = 1.0f;
	if (current_health <= LowHealthThreshold)
	{
		if (current_time < EndBlinkTime)
		{
			blink_percentage = float(EndBlinkTime - current_time) / float (LowHealthBlinkDuration);
		}
		else
		{
			EndBlinkTime = current_time + LowHealthBlinkDuration;
		}
	};

	Vector4 background_color;
	background_color.X = 1.0f; // r
	background_color.Y = 1.0f; // g
	background_color.Z = 1.0f; // b
	background_color.W = flash_percentage; // a

	Vector4 text_color = Vector4(1.0f, 1.0f, 1.0f, 1.0f);
	text_color.X = 1.0f;
	text_color.Y = blink_percentage;
	text_color.Z = blink_percentage; 
	text_color.W = flash_percentage;

	Render2D->Reset();
	Render2D->Shader.ShaderBits = 0x000984B3;

	Render2D->Set_Texture(BackgroundTextureFile);
	if (Render2D->Texture->Initialized == false)
	{
		Render2D->Texture->Init();
	};

	RectClass background_location;
	background_location.Top = BackgroundPosition.Y;
	background_location.Left = BackgroundPosition.X;
	background_location.Bottom = BackgroundPosition.Y + BackgroundSize.Y;
	background_location.Right = BackgroundPosition.X + BackgroundSize.X;

	Render2D->Add_Quad(background_location, BackgroundUV, Vector4ToD3DCOLOR(background_color));
	Render2D->Render();

	Render2DText->Reset();
	RectClass *r = (RectClass *)((char *)Render2DText+0x5B8); // don't try this at home
	r->Top = TextPosition.Y; 
	r->Left = TextPosition.X; 
	r->Bottom = TextPosition.Y; 
	r->Right = TextPosition.X;
	char temp[32];
	sprintf(temp, "%d", (int)current_health),
	Render2DText->Draw_Text(temp, Vector4ToD3DCOLOR(text_color));
	Render2DText->Render();

	LastHealth = current_health;
};

//--------------------------------------------------------------------------------
// globals
//--------------------------------------------------------------------------------
MiniHealthHUDItemClass MiniHealthHUD;