// THUHandler.cpp - Implementation of the CTHUHandler class
//	Handles THU files and can parse the contents
// Kevin Kirst
// kkirst@fullsail.edu

#include "THUHandler.h"

// Constructor
CTHUHandler::CTHUHandler(void)
{
	// Remove the content
	memset(&this->m_hHeader, 0, sizeof(THUHEADER));
	this->m_Textures.empty();
	this->m_pTextureDataSection = NULL;
}

// Constructor
CTHUHandler::CTHUHandler(char const* filepath)
{
	// Remove the content
	memset(&this->m_hHeader, 0, sizeof(THUHEADER));
	this->m_Textures.empty();
	this->m_pTextureDataSection = NULL;

	// Load
	this->Load(filepath);
}

// Destructor
CTHUHandler::~CTHUHandler(void)
{
	this->Unload();
}

// Load - Load a THU file
//
// In:	filepath	File path of THU file to load
//		bOverwrite	Set to TRUE to overwrite the file
//
// Returns TRUE on success
bool CTHUHandler::Load(char const* filepath, bool bOverwrite /*= false */)
{
	// Fail if a file is already loaded
	if (true == this->m_FileStream.is_open()) return false;

	// Attempt to open the file
	std::ios_base::open_mode flags = std::ios::in | std::ios::out | std::ios::binary;
	if (true == bOverwrite) flags |= std::ios::trunc;
	this->m_FileStream.open(filepath, flags);
	if (false == this->m_FileStream.is_open()) return false;

	// Extract the data
	this->_ParseTHUFile();

	return true;
}

// Update - Update the loaded THU file with the edited content contained in the object
//
// Returns TRUE on success
bool CTHUHandler::Update(void)
{
	// Fail if no file is loaded
	if (false == this->m_FileStream.is_open()) return false;

	// Update it!
	this->_RewriteTHUFile();
	return true;
}

// Unload - Unload the loaded THU file
void CTHUHandler::Unload(void)
{
	// Remove the content
	memset(&this->m_hHeader, 0, sizeof(THUHEADER));
	this->m_Textures.empty();
	delete [] this->m_pTextureDataSection;
	this->m_pTextureDataSection = NULL;

	this->m_FileStream.close();
}

// _ParseTHUFile - Physically parse the THU file and extract the content
void CTHUHandler::_ParseTHUFile(void)
{
	if (false == this->m_FileStream.is_open()) return; // Must be opened!

	// Remove the content
	memset(&this->m_hHeader, 0, sizeof(THUHEADER));
	this->m_Textures.empty();
	delete [] this->m_pTextureDataSection;
	this->m_pTextureDataSection = NULL;

	// Read the main header
	this->m_FileStream.seekg(0, std::ios::beg);
	this->m_FileStream.read((char*)&this->m_hHeader, sizeof(THUHEADER));

	// Read in the texture elements
	DWORD dwTextureCount = this->m_hHeader.dwTextureCount;
	THUTEXTURE hThisTexture;
	this->m_Textures.reserve(dwTextureCount);
	for (DWORD txtnum = 0; txtnum < dwTextureCount; txtnum++)
	{
		memset(&hThisTexture, 0, sizeof(THUTEXTURE));

		// Read this texture in
		this->m_FileStream.read((char*)&hThisTexture.dwIdentifier, sizeof(DWORD));
		this->m_FileStream.read((char*)&hThisTexture.dwDataOffset, sizeof(DWORD));
		this->m_FileStream.read((char*)&hThisTexture.dwUnknown1, sizeof(DWORD));
		this->m_FileStream.read((char*)&hThisTexture.dwUnknown2, sizeof(DWORD));
		this->m_FileStream.read((char*)&hThisTexture.dwWidth, sizeof(DWORD));
		this->m_FileStream.read((char*)&hThisTexture.dwHeight, sizeof(DWORD));
		this->m_FileStream.read((char*)&hThisTexture.dwDimMult, sizeof(DWORD));
		this->m_FileStream.read((char*)&hThisTexture.dwPixelFormat, sizeof(DWORD));
		this->m_FileStream.read((char*)&hThisTexture.dwFileLength, sizeof(DWORD));
		this->m_FileStream.read((char*)&hThisTexture.pchFile, hThisTexture.dwFileLength);

		// Put it in the list
		this->m_Textures.push_back(hThisTexture);
	}

	// Read in the texture data content
	this->m_pTextureDataSection = new char [this->m_hHeader.dwUnkSecSize];
	this->m_FileStream.read(this->m_pTextureDataSection, this->m_hHeader.dwUnkSecSize);
}

// _RewriteTHUFile - Rewrite the THU file using updated info in class object
void CTHUHandler::_RewriteTHUFile(void)
{
	// Seek to the beginning and write out the header
	this->m_FileStream.seekp(0, std::ios::beg);
	this->m_FileStream.write((char*)&this->m_hHeader, sizeof(THUHEADER));

	// Write out the texture elements
	for (TEXTURES::iterator i = this->m_Textures.begin(); i != this->m_Textures.end(); i++)
	{
		this->m_FileStream.write((char*)&i->dwIdentifier, sizeof(DWORD));
		this->m_FileStream.write((char*)&i->dwDataOffset, sizeof(DWORD));
		this->m_FileStream.write((char*)&i->dwUnknown1, sizeof(DWORD));
		this->m_FileStream.write((char*)&i->dwUnknown2, sizeof(DWORD));
		this->m_FileStream.write((char*)&i->dwWidth, sizeof(DWORD));
		this->m_FileStream.write((char*)&i->dwHeight, sizeof(DWORD));
		this->m_FileStream.write((char*)&i->dwDimMult, sizeof(DWORD));
		this->m_FileStream.write((char*)&i->dwPixelFormat, sizeof(DWORD));
		this->m_FileStream.write((char*)&i->dwFileLength, sizeof(DWORD));
		this->m_FileStream.write((char*)&i->pchFile, i->dwFileLength);
	}

	// Write out the texture data
	this->m_FileStream.write(this->m_pTextureDataSection, this->m_hHeader.dwUnkSecSize);
}