// DXSystem.cpp - Implementation of the CDXSystem class
//	Handles basic initilization / releation of DX controls
// Kevin Kirst
// kkirst@fullsail.edu

#include <ctime>
#include "DXSystem.h"
#include "DXCamera.h"
#include "DXViewFrustum.h"
#include "DXInput.h"
#include "DXSoundManager.h"
#include "DXEntity.h"
#include "DXSprite.h"

#define INPSND_UPDATE_TIMER		0.016f

CDXSystem &DXSystem = CDXSystem::GetInstance();

// Constructor
CDXSystem::CDXSystem(void)
{
	this->m_pD3DObject = NULL;
	this->m_pD3DDevice = NULL;
	this->m_pD3DSpriteMngr = NULL;
	this->m_hViewPort.pWnd = NULL;
	this->m_pCamera = NULL;
	this->m_pVF = NULL;
	this->m_pInputSys = NULL;
	this->m_pSoundMngr = NULL;
	this->m_bLighting = false;

	this->m_nFPS = this->m_nFrameCount = 0;
    QueryPerformanceFrequency(&this->m_nTickFrequency);
	LARGE_INTEGER nTicks;
    QueryPerformanceCounter(&nTicks);
    this->m_fTimeStart = (float)nTicks.QuadPart/(float)this->m_nTickFrequency.QuadPart;
	this->m_fTimeDelta = 0.0f;

	//randomize!
	srand(static_cast<unsigned int>(time(NULL)));
}

// Destructor
CDXSystem::~CDXSystem(void)
{
	delete this->m_pCamera;
	delete this->m_pVF;
	delete this->m_pInputSys;
	delete this->m_pSoundMngr;

	// Safely release device and object
	if (NULL != this->m_pD3DSpriteMngr)
	{
		this->m_pD3DSpriteMngr->Release();
		this->m_pD3DSpriteMngr = NULL;
	}
	if (NULL != this->m_pD3DDevice)
	{
		this->m_pD3DDevice->Release();
		this->m_pD3DDevice = NULL;
	}
	if (NULL != this->m_pD3DObject)
	{
		this->m_pD3DObject->Release();
		this->m_pD3DObject = NULL;
	}
}

// Init - Initialize DX
//
// In:	hVP		HVIEWPORT describing the viewport
//		wndmode	(TRUE) Set to TRUE to run in windowed mode;
//				FALSE for fullscreen
//
// Returns TRUE if everything was setup A-OK
bool CDXSystem::Init(HVIEWPORT &hVP, bool wndmode /*=true*/)
{
	this->m_hViewPort = hVP;

	// Create the object
	if (NULL == (this->m_pD3DObject = Direct3DCreate9(D3D_SDK_VERSION))) return false;
	
	// Create the device
	D3DPRESENT_PARAMETERS D3PP;
	ZeroMemory(&D3PP, sizeof(D3DPRESENT_PARAMETERS));
	D3PP.Windowed = wndmode;														// Windowed mode?
	D3PP.BackBufferFormat = true == wndmode ? D3DFMT_UNKNOWN : D3DFMT_X8R8G8B8;		// Backbuffer format
	D3PP.BackBufferCount = 1;														// Backbuffer count
	D3PP.BackBufferWidth = m_hViewPort.nWidth;
	D3PP.BackBufferHeight = m_hViewPort.nHeight;
	D3PP.SwapEffect = D3DSWAPEFFECT_DISCARD;										// Discard buffer after use
	D3PP.PresentationInterval = D3DPRESENT_INTERVAL_IMMEDIATE;						// Show changes immediately
	D3PP.EnableAutoDepthStencil = true;
	D3PP.AutoDepthStencilFormat = D3DFMT_D16;
	
	if (FAILED(this->m_pD3DObject->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_REF, *this->m_hViewPort.pWnd,
		D3DCREATE_HARDWARE_VERTEXPROCESSING, &D3PP, &this->m_pD3DDevice))) return false;

	this->m_pD3DDevice->SetRenderState(D3DRS_ZENABLE, D3DZB_TRUE);

	// Create the sprite manager
	if (FAILED(D3DXCreateSprite(this->m_pD3DDevice, &this->m_pD3DSpriteMngr))) return false;

	// Setup the camera
	this->m_pCamera = new CDXCamera;
	if (NULL == this->m_pCamera) return false;

	// Setup the view Frustum
	this->m_pVF = new CDXViewFrustum;
	if (NULL == this->m_pVF) return false;

	// Initialize input control
	this->m_pInputSys = new CDXInput;
	if (NULL == this->m_pInputSys) return false;
	this->m_pInputSys->Init(&this->m_hViewPort);

	// Setup the sound manager
	this->m_pSoundMngr = new CDXSoundManager;
	if (NULL == this->m_pSoundMngr) return false;
	this->m_pSoundMngr->Init(&this->m_hViewPort);

	// Everything is initialized
	return true;	
}

// Update - Update loaded entities, sprites, input/sound queues
void CDXSystem::Update(void)
{
	// Handle timer-related updates first
    LARGE_INTEGER nTicks;
    QueryPerformanceCounter(&nTicks);
    this->m_fTimeNow = (float)nTicks.QuadPart/(float)this->m_nTickFrequency.QuadPart - this->m_fTimeStart;
	static float fTimePrev = this->m_fTimeNow;
	static float fTimeLastUpdate = this->m_fTimeNow;
	static float fTimeLastInputUpdate = this->m_fTimeNow;

    // Handle new FPS count if a second has ellapsed
    if (this->m_fTimeNow - fTimePrev > 1.0f)
    {
        this->m_nFPS = this->m_nFrameCount / (int)(this->m_fTimeNow - fTimePrev);
        this->m_nFrameCount = 0;
        fTimePrev = this->m_fTimeNow;
    }

    // Caulculate dt
    this->m_fTimeDelta = this->m_fTimeNow - fTimeLastUpdate;
    fTimeLastUpdate = this->m_fTimeNow;

	if (this->m_fTimeNow - fTimeLastInputUpdate >= INPSND_UPDATE_TIMER)
	{
		// Update input every other frame
		if (NULL != this->m_pInputSys)
			this->m_pInputSys->Update(this->m_fTimeDelta);

		// Update sounds every other frame
		if (NULL != this->m_pSoundMngr)
			this->m_pSoundMngr->Update(this->m_fTimeDelta);
	}

	// Update all entities
	if (this->m_hDrawList.size() > 0)
	{
		for (ENTITYLIST::iterator i = this->m_hDrawList.begin(); i != this->m_hDrawList.end(); i++)
		{
			(*i)->Update(this->m_fTimeDelta);
		}
	}

	// Update all sprites
	if (this->m_hSpriteList.size() > 0)
	{
		for (SPRITELIST::iterator i = this->m_hSpriteList.begin(); i != this->m_hSpriteList.end(); i++)
		{
			(*i)->Update(this->m_fTimeDelta);
		}
	}
}

// Render - Render loaded entities and sprites
//
// In:	bClear	(TRUE) Set to TRUE to clear the screen
void CDXSystem::Render(bool bClear /*= TRUE*/)
{
	if (NULL == this->m_pD3DDevice) return; // Can't draw without a device!

	// Clear the screen
	if (true == bClear)
	{
		this->m_pD3DDevice->Clear(0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER,
			D3DCOLOR_XRGB(0,0,0), 1.0f, 0);
	}

	// Setup the camera
	this->m_pD3DDevice->SetTransform(D3DTS_VIEW,
		&this->m_pCamera->CalculateViewMatrix());

	// Setup projection
	this->m_pD3DDevice->SetTransform(D3DTS_PROJECTION,
		&this->m_pVF->CalculateProjectionMatrix());

	// Start the scene
	if ((this->m_hDrawList.size() > 0 || this->m_hSpriteList.size() > 0) &&
		SUCCEEDED(this->m_pD3DDevice->BeginScene()))
	{
		// Draw the entities
		for (ENTITYLIST::iterator i = this->m_hDrawList.begin(); i != this->m_hDrawList.end(); i++)
		{
			// Only draw visible entities
			if (false == (*i)->IsVisible()) continue;
			
			this->m_pD3DDevice->SetTransform(D3DTS_WORLD, &(*i)->CalcWorldMatrix());

			// Render it
			(*i)->PreRender(this->m_pD3DDevice);
			if (true == this->m_bLighting) // Set master lighting on if it is turned on
				this->m_pD3DDevice->SetRenderState(D3DRS_LIGHTING, false);
			(*i)->Render(this->m_pD3DDevice);
			(*i)->PostRender(this->m_pD3DDevice);
		}

		// Draw the sprites
		if (SUCCEEDED(this->m_pD3DSpriteMngr->Begin(D3DXSPRITE_ALPHABLEND | D3DXSPRITE_SORT_DEPTH_FRONTTOBACK)))
		{
			// Draw the sprites
			for (SPRITELIST::iterator i = this->m_hSpriteList.begin(); i != this->m_hSpriteList.end(); i++)
			{
				// Render it
				(*i)->Render(this->m_pD3DSpriteMngr);
			}

			this->m_pD3DSpriteMngr->End();
		}

		// End the scene
		this->m_pD3DDevice->EndScene();
	}

	this->m_nFrameCount++;
}

// Present - Preset the rendered scene
void CDXSystem::Present(void)
{
	// Present it!
	this->m_pD3DDevice->Present(NULL, NULL, NULL, NULL);
}

// AddEntityToDrawList - Add the specified entity to the drawlist
//
// In:	pEntity		Entity to add
void CDXSystem::AddEntityToDrawList(CDXEntity *pEntity)
{
	// Add the entity if it already isn't in there
	for (ENTITYLIST::iterator i = this->m_hDrawList.begin(); i != this->m_hDrawList.end(); i++)
	{
		if (*i == pEntity) return;
	}
	this->m_hDrawList.push_back(pEntity);
}

// RemoveEntityFromDrawList - Remove the specied entity from the drawlist
//
// In:	pEntity		Entity to remove
void CDXSystem::RemoveEntityFromDrawList(CDXEntity *pEntity)
{
	// Find it
	for (ENTITYLIST::iterator i = this->m_hDrawList.begin(); i != this->m_hDrawList.end(); i++)
	{
		if (*i == pEntity)
		{
			this->m_hDrawList.erase(i);
			return;
		}
	}
}

// AddSpriteToDrawList - Add the specified sprite to the sprite drawlist
//
// In:	pSprite		Sprite to add
void CDXSystem::AddSpriteToDrawList(CDXSprite *pSprite)
{
	// Add the entity if it already isn't in there
	for (SPRITELIST::iterator i = this->m_hSpriteList.begin(); i != this->m_hSpriteList.end(); i++)
	{
		if (*i == pSprite) return;
	}
	this->m_hSpriteList.push_back(pSprite);
}

// RemoveSpriteFromDrawList - Remove the specied sprite from the sprite drawlist
//
// In:	pSprite		Sprite to remove
void CDXSystem::RemoveSpriteFromDrawList(CDXSprite *pSprite)
{
	// Find it
	for (SPRITELIST::iterator i = this->m_hSpriteList.begin(); i != this->m_hSpriteList.end(); i++)
	{
		if (*i == pSprite)
		{
			this->m_hSpriteList.erase(i);
			return;
		}
	}
}